﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common.Extensions;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapLicensureToCrm : MapperRelatedBase
    {
        public MapLicensureToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_providerlicensure> MapInsert(Licensure licensure, Account provider)
        {
            // map entity
            var entity = new ppms_providerlicensure
            {
                Id = Guid.NewGuid(),
                ppms_name = licensure.LicenseNumber,
                ppms_licensingstate = licensure.LicensingState
            };

            if (licensure.ExpirationDateSpecified && licensure.ExpirationDate.Year > EnumHelper.MinYear) entity.ppms_expirationdate = licensure.ExpirationDate.ToCrmDate();

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => {});

            return entity;
        }

        public async Task<ppms_providerlicensure> MapUpdate(Licensure entity, Account provider)
        {
            // get entity
            var crmEntity = GetCrmEntity(provider, entity.LicenseNumber);

            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            var newEntity = new ppms_providerlicensure()
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name,
                ppms_licensingstate = crmEntity.ppms_licensingstate,
                ppms_expirationdate = crmEntity.ppms_expirationdate
            };

            //Map fields
            if (IsChanged(entity.LicenseNumber, newEntity.ppms_name))
            {
                newEntity.ppms_name = entity.LicenseNumber;
            }

            if (IsChanged(entity.LicensingState, newEntity.ppms_licensingstate))
            {
                newEntity.ppms_licensingstate = entity.LicensingState;
            }

            if (entity.ExpirationDateSpecified && entity.ExpirationDate != newEntity.ppms_expirationdate && entity.ExpirationDate.Year > EnumHelper.MinYear)
            {
                newEntity.ppms_expirationdate = entity.ExpirationDate.ToCrmDate();
            }

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            await Task.Run(() => {});

            return newEntity;
        }

        private static ppms_providerlicensure GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_providerlicensure;
            if (list == null) return null;

            var ppmsProviderlicensures = list as ppms_providerlicensure[] ?? list.ToArray();

            return ppmsProviderlicensures.Any() ? ppmsProviderlicensures.FirstOrDefault(x => x.ppms_name == id) : null;
        }

        private static Licensure ConvertEntity<T>(T entity)
        {
            return (Licensure)Convert.ChangeType(entity, typeof(Licensure));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_providerlicensure");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null) account.ppms_account_providerlicensure = ConvertEntityList<ppms_providerlicensure>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_providerlicensure>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var matches = provider.ppms_account_providerlicensure.Where(
                    p => p.ppms_name == entity.LicenseNumber &&
                    p.ppms_licensingstate == entity.LicensingState &&
                    (!entity.ExpirationDateSpecified || p.ppms_expirationdate == entity.ExpirationDate)
                );
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_providerlicensure>)list);
        }
    }
}